%-------------------------------------------------------------------------%
% Copyright (c) 2017 Modenese L.                                          %
%                                                                         %
% Licensed under the Apache License, Version 2.0 (the "License");         %
% you may not use this file except in compliance with the License.        %
% You may obtain a copy of the License at                                 %
% http://www.apache.org/licenses/LICENSE-2.0.                             %
%                                                                         %
% Unless required by applicable law or agreed to in writing, software     %
% distributed under the License is distributed on an "AS IS" BASIS,       %
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or         %
% implied. See the License for the specific language governing            %
% permissions and limitations under the License.                          %
%                                                                         %
%    Author:   Luca Modenese,  2017                                       %
%    email:    l.modenese08@imperial.ac.uk                                %
% ----------------------------------------------------------------------- %
% This script allows to load an OpenSim musculoskeletal model using the
% application programming interface (API) and decreases the strength of the
% muscles crossing the knee by 35% as done for instance in Marra
% et al (2015) Journal of Biomechanical Engineering 137, 020904 in order to
% model the muscle strength reduction following total knee replacement.
%
% NOTE: More examples of Matlab scripting are availablein the folder OpenSim 3.3\Scripts 
%
% ----------------------------------------------------------------------- %
% Last modified: 10 Oct 2017


%---------- USER'S SETTINGS -----------
% specify the name (and path, if necessary) of the OpenSim model to modify
osimModel_name = 'workshopModel2392-scaled.osim';

% specify the muscles for which maximum isometric force will be modified
% in the workshop example, prothesis is on the left side.
updMuscles_set = {'sar_l','semimem_l','semiten_l','bifemlh_l','bifemsh_l',...
                  'tfl_l','grac_l','rect_fem_l','vas_med_l','vas_int_l',...
                  'vas_lat_l'};

% specify the scaling factor for the maximum isometric force (strength) of
% the muscles in the list above.
strength_scaling_factor = 0.65;%
%--------------------------------------

% import OpenSim libraries
import org.opensim.modeling.*

% load model
osimModel = Model(osimModel_name);

% get muscles
muscles = osimModel.getMuscles();

% number of muscles to modify
N = size(updMuscles_set,2);

% loop through the desired muscles 
for n_mus = 1:N
    % get name of muscle to modify
    cur_mus_name = updMuscles_set{n_mus};
    
    % get muscles to modify
    cur_mus = muscles.get(cur_mus_name);
    
    % compute new maximum isometric force
    Fiso = cur_mus.getMaxIsometricForce();
    updFiso = Fiso*strength_scaling_factor;
    
    % update maximum isometric force
    cur_mus.setMaxIsometricForce(updFiso);
    
    % inform user
    display(['Muscle ',cur_mus_name,': Max isom force modified from ',num2str(Fiso),' to ',num2str(updFiso)]);
end

% write model with modified muscles
[p,n,e] = fileparts(osimModel_name);
osimModel.print(fullfile(p,[n,'_updFisoKneeMuscles',e]));

